/*
This header implements the bresenham's drawing algorithms for lines
and circles.

  Michael Chungkun Chen
  002-764-157
*/
#ifndef BRES
#define BRES

#include "include.h"    //Header files for GL, for windows and unix
#include "objtypes.h"   //Structures used for the objects

void Bres_Line( int X0, int Y0, int X1, int Y1 )
/*************************************************************************
This function uses the Bresenham Line drawing algorithm to rasterize
a line on the screen given two points.
**************************************************************************/
{
  int d,           //Decision variable for Bressenham's algorithm
      A, B,        //Part of Ax+By+C=0
      x, y,        //x and y coordinates of current pixel to turn on.
      s,           //Sign, controls adding or subtracting for
                   //positive/negative slopes.

      inca,incb,   //Increment variables to increase efficiency
      leftx,rightx,//Variable used in order to figure
      lefty,righty;//   out which point in on the left
  
  if (X0<X1) //Determine the left point and right point
    {
      leftx=X0;
      rightx=X1;
      lefty=Y0;
      righty=Y1;
    }
  else
    {
      leftx=X1;
      rightx=X0;
      lefty=Y1;
      righty=Y0;
    }
  B=rightx-leftx;  //Calculate B delta X
  A=righty-lefty;  //Calculate A delta y
  
  if (abs(B)>abs(A))  //Slope is -45<=m<=45 degrees
    {
      x=leftx;	    //Start at the left most point
      y=lefty;
      
     if (A<0)       //Setup the add/subtract control based on the
       s=-1;        //sign of A
     else
       s=1;
     
     d=2*A*s-2*B-1; //Setup the initial decision variable
     inca=2*s*A;    //Setup increment for going east
     incb=2*A*s-2*B;//Setup increment for north east
     
     glRasterPos2i( x, y );//Places pixel on raster (RED)
     glDrawPixels( 1, 1, GL_RGBA, GL_FLOAT, ( GLvoid * ) color );
     
     while (x<rightx) //Repeat until endpoint reached.
       {
      	 if (d<0)	  //Choose east
      	   {
	           x++;
      	     d+=inca;
	         }
	       else		  //Choose (north/south)east
	         {
	           x++;
      	     y+=s;
      	     d+=incb;
           }
      	 glRasterPos2i( x, y );//Places pixel on raster (RED)
      	 glDrawPixels( 1, 1, GL_RGBA, GL_FLOAT, ( GLvoid * ) color );
       }//end while
    }
  else if (abs(B)<abs(A))
  //We have a line m<-45 degrees or m>45 degrees
    {
      if (Y0<Y1) //Refigure out bottom most point with X, Y flipped.
     	{          //Bottom translates to left.  Top to right.
    	  leftx=Y0;
    	  rightx=Y1;
    	  lefty=X0;
    	  righty=X1;
    	}
      else
    	{
    	  leftx=Y1;
	      rightx=Y0;
	      lefty=X1;
    	  righty=X0;
	    }
      
      B=rightx-leftx; //Figure out B and A
      A=righty-lefty;
      x=leftx;      //Initialize starting point
      y=lefty;
      if (A<0)      //Determine positive/negative slope
      	s=-1;
      else
	      s=1;
      
      d=2*A*s-2*B-1;//Initialize decision variable
      
      inca=2*s*A;   //Increment varible for north
      incb=2*A*s-2*B; //Increment variable for north(west/east)
      glRasterPos2i( y, x );//Draw pixel
      glDrawPixels( 1, 1, GL_RGBA, GL_FLOAT, ( GLvoid * ) color );
      
      while (x<rightx) //Loop until end is reached
    	{
    	  if (d<0)	  //Go north
	      {
	        x++;
	        d+=inca;
	      }
	      else          //Go north(east/west)
	      {
	        x++;
	        y+=s;
	        d+=incb;
	      }
	      glRasterPos2i( y, x );//Draw pixel
	      glDrawPixels( 1, 1, GL_RGBA, GL_FLOAT, ( GLvoid * ) color ); 
	    }//end while
      
    }
  else
  {
     x=leftx;	    //Start at the left most point
     y=lefty;
      
     if (A<0)       //Setup the add/subtract control based on the
       s=-1;        //sign of A
     else
       s=1;
      
     glRasterPos2i( x, y );//Places pixel on raster (RED)
     glDrawPixels( 1, 1, GL_RGBA, GL_FLOAT, ( GLvoid * ) color );
     
     while (x<rightx) //Repeat until endpoint reached.
       {
           x++;
           y+=s;
      	 glRasterPos2i( x, y );//Places pixel on raster (RED)
      	 glDrawPixels( 1, 1, GL_RGBA, GL_FLOAT, ( GLvoid * ) color );
       }//end while
    }

#ifndef fast
  glFlush();     //Flushes the opengl buffer to execute the queued up instructions.
#endif
  return;
}

void Bres_Circle( int X, int Y, int R)
/*************************************************************************
This function implements the Bressenham algorithm to draw a circle.
It eliminates floating point math, and is highly optimized for speed.

The concept for the circle, is to draw a circle assuming (0,0) to be the
center, when the pixels are actually placed on the screen, the center offset
X, and Y are added to the position.
**************************************************************************/
{
  int incx, incy;  //Increment variables
  int d;	   //Decision variable
  int x, y;	   //x,y coords relative to circle's center.
  
  //Initialize x and y coordinates
  x=0;
  y=R;
  
  //Draws four pixels where circle would intersect axis.
  glRasterPos2i( X+x, Y+y );//Draw pixel
  glDrawPixels( 1, 1, GL_RGBA, GL_FLOAT, ( GLvoid * ) color );
  glRasterPos2i( X+x, Y-y );//Draw pixel
  glDrawPixels( 1, 1, GL_RGBA, GL_FLOAT, ( GLvoid * ) color );
  glRasterPos2i( X-y, Y+x );//Draw pixel
  glDrawPixels( 1, 1, GL_RGBA, GL_FLOAT, ( GLvoid * ) color );
  glRasterPos2i( X+y, Y+x );//Draw pixel
  glDrawPixels( 1, 1, GL_RGBA, GL_FLOAT, ( GLvoid * ) color );
  
  //Setup initial decision variable.  Instead of 5/4 we use 2
  d=2-R;
  
  //Setup the increment variable
  incx=3;
  incy=-2*R+2;
  
  //Loops until one octant is drawn.
  while (y>x)
    {
      
      d+=incx;	  //Update decision	
      if (d>=0) //If choosing southeast
	{
	  d+=incy;  //Update decision some more
	  incy+=2;  //Update increment for y only here.
	  y--;	  //Move current pixel location
	}
      
      x++;		  //Move pixel location
      incx+=2;	  //Update increment for x
      
      //The following places pixels in the eight respective octants.
      //O# means the octant the instruction would put the pixel in.
      //CW = Clockwise      CCW = CounterClockwise
      glRasterPos2i( X+x, Y+y );//Draw pixel  O2  CW
      glDrawPixels( 1, 1, GL_RGBA, GL_FLOAT, ( GLvoid * ) color );
      glRasterPos2i( X-x, Y+y );//Draw pixel  O3  CCW
      glDrawPixels( 1, 1, GL_RGBA, GL_FLOAT, ( GLvoid * ) color );
      glRasterPos2i( X+x, Y-y );//Draw pixel  O7  CCW
      glDrawPixels( 1, 1, GL_RGBA, GL_FLOAT, ( GLvoid * ) color );
      glRasterPos2i( X-x, Y-y );//Draw pixel  O6  CW
      glDrawPixels( 1, 1, GL_RGBA, GL_FLOAT, ( GLvoid * ) color );
      glRasterPos2i( X+y, Y+x );//Draw pixel  O1  CCW
      glDrawPixels( 1, 1, GL_RGBA, GL_FLOAT, ( GLvoid * ) color );
      glRasterPos2i( X+y, Y-x );//Draw pixel  O8  CW
      glDrawPixels( 1, 1, GL_RGBA, GL_FLOAT, ( GLvoid * ) color );
      glRasterPos2i( X-y, Y+x );//Draw pixel  O4  CW
      glDrawPixels( 1, 1, GL_RGBA, GL_FLOAT, ( GLvoid * ) color );
      glRasterPos2i( X-y, Y-x );//Draw pixel  O5  CCW
      glDrawPixels( 1, 1, GL_RGBA, GL_FLOAT, ( GLvoid * ) color );		
      //Pixels placed
    }//End of loop
#ifndef fast
  glFlush(); //Flush GL buffer
#endif
  return;
}

#endif
